# 测试日志记录和文件操作中的异常处理

import logging
import os

# 配置日志
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler("exception_log.txt"),
        logging.StreamHandler()
    ]
)

def safe_file_operations():
    """安全的文件操作示例"""
    filename = "test_file.txt"
    
    # 写入文件
    try:
        with open(filename, 'w', encoding='utf-8') as f:
            f.write("这是测试文件的内容\n")
            f.write("第二行内容")
        logging.info(f"文件 {filename} 写入成功")
    except PermissionError:
        logging.error(f"没有权限写入文件: {filename}")
        return
    except OSError as e:
        logging.error(f"写入文件时发生系统错误: {e}")
        return
    
    # 读取文件
    try:
        with open(filename, 'r', encoding='utf-8') as f:
            content = f.read()
            logging.info(f"成功读取文件 {filename}")
            print(f"文件内容:\n{content}")
    except FileNotFoundError:
        logging.error(f"文件不存在: {filename}")
    except PermissionError:
        logging.error(f"没有权限读取文件: {filename}")
    except UnicodeDecodeError:
        logging.error(f"文件编码错误: {filename}")
    except Exception as e:
        logging.error(f"读取文件时发生未知错误: {e}")
    finally:
        # 清理工作（如果需要）
        logging.info("文件操作完成")

def process_multiple_files(file_list):
    """处理多个文件，演示异常处理和日志记录"""
    successful_files = 0
    failed_files = 0
    
    for filename in file_list:
        try:
            logging.info(f"开始处理文件: {filename}")
            
            # 检查文件是否存在
            if not os.path.exists(filename):
                raise FileNotFoundError(f"文件 {filename} 不存在")
            
            # 读取文件内容
            with open(filename, 'r', encoding='utf-8') as f:
                content = f.read()
                
            # 处理文件内容（这里只是简单地计算行数）
            line_count = len(content.split('\n'))
            logging.info(f"文件 {filename} 包含 {line_count} 行")
            
            # 写入处理结果
            result_filename = f"{filename}.result"
            with open(result_filename, 'w', encoding='utf-8') as f:
                f.write(f"原始文件: {filename}\n")
                f.write(f"行数: {line_count}\n")
                
            successful_files += 1
            logging.info(f"文件 {filename} 处理成功")
            
        except FileNotFoundError as e:
            logging.error(f"文件未找到: {e}")
            failed_files += 1
        except PermissionError as e:
            logging.error(f"权限错误: {e}")
            failed_files += 1
        except UnicodeDecodeError as e:
            logging.error(f"编码错误: {e}")
            failed_files += 1
        except Exception as e:
            logging.error(f"处理文件 {filename} 时发生未知错误: {e}")
            failed_files += 1
    
    # 返回处理结果统计
    return successful_files, failed_files

if __name__ == "__main__":
    print("1. 安全的文件操作示例")
    safe_file_operations()
    
    print("\n2. 处理多个文件示例")
    # 准备测试文件列表（包括存在和不存在的文件）
    test_files = [
        "test_file.txt",           # 存在（由上面的函数创建）
        "不存在的文件.txt",          # 不存在
        "test_file.txt.result"     # 存在（由上面的函数创建）
    ]
    
    successful, failed = process_multiple_files(test_files)
    print(f"处理结果: {successful} 个文件成功, {failed} 个文件失败")
    print("详细日志已写入 exception_log.txt 文件")