# 测试异常处理的各种情况

# 1. 基本异常捕获
def test_basic_exception():
    print("\n1. 基本异常捕获示例")
    try:
        x = 1 / 0
    except ZeroDivisionError as e:
        print(f"捕获到除零异常: {e}")

# 2. 捕获多种异常
def test_multiple_exceptions():
    print("\n2. 捕获多种异常示例")
    # 使用多个except块
    try:
        # 尝试访问不存在的列表索引
        my_list = [1, 2, 3]
        print(my_list[10])
    except IndexError as e:
        print(f"索引错误: {e}")
    except TypeError as e:
        print(f"类型错误: {e}")
    
    # 使用元组形式捕获多种异常
    try:
        # 尝试将字符串转换为整数
        int("abc")
    except (ValueError, TypeError) as e:
        print(f"值错误或类型错误: {e}")

# 3. finally子句
def test_finally():
    print("\n3. finally子句示例")
    try:
        print("尝试打开一个不存在的文件")
        with open("不存在的文件.txt", "r") as f:
            content = f.read()
    except FileNotFoundError as e:
        print(f"文件未找到: {e}")
    finally:
        print("无论是否发生异常，finally块都会执行")

# 4. else子句
def test_else():
    print("\n4. else子句示例")
    try:
        x = 10
        y = 2
        result = x / y
    except ZeroDivisionError:
        print("除数不能为零!")
    else:
        print(f"计算结果: {result}")

# 5. raise语句
def test_raise():
    print("\n5. raise语句示例")
    try:
        x = -5
        if x < 0:
            raise ValueError("不能使用负数")
    except ValueError as e:
        print(f"捕获到值错误: {e}")

# 6. 异常链
def test_exception_chaining():
    print("\n6. 异常链示例")
    try:
        try:
            1 / 0
        except ZeroDivisionError as e:
            # 引发新异常，保留原始异常信息
            raise ValueError("计算过程中出现错误") from e
    except ValueError as e:
        print(f"捕获到值错误: {e}")
        # 打印异常链信息
        if e.__cause__:
            print(f"原始异常: {e.__cause__}")

# 7. 自定义异常
class MyCustomError(Exception):
    """自定义异常类"""
    def __init__(self, message, error_code=None):
        self.message = message
        self.error_code = error_code
        super().__init__(self.message)
    
    def __str__(self):
        if self.error_code:
            return f"{self.message} (错误代码: {self.error_code})"
        return self.message

def test_custom_exception():
    print("\n7. 自定义异常示例")
    try:
        raise MyCustomError("这是一个自定义异常", 1001)
    except MyCustomError as e:
        print(f"捕获到自定义异常: {e}")
        print(f"错误代码: {e.error_code}")

# 8. 捕获所有异常
def test_catch_all():
    print("\n8. 捕获所有异常示例")
    try:
        # 尝试执行一些可能出错的代码
        x = undefined_variable  # 未定义的变量
    except Exception as e:
        print(f"捕获到异常: {type(e).__name__}: {e}")

# 执行测试函数
if __name__ == "__main__":
    test_basic_exception()
    test_multiple_exceptions()
    test_finally()
    test_else()
    test_raise()
    test_exception_chaining()
    test_custom_exception()
    test_catch_all()