# 测试异常层次结构和编程模式

# 1. 异常层次结构演示
def test_exception_hierarchy():
    print("1. 异常层次结构演示")
    
    # 显示常见异常的继承关系
    exceptions = [
        ZeroDivisionError(),
        ValueError(),
        TypeError(),
        IndexError(),
        KeyError(),
        FileNotFoundError(),
        AttributeError(),
        ImportError()
    ]
    
    for exc in exceptions:
        print(f"{type(exc).__name__} -> {type(exc).__bases__[0].__name__}")

# 2. EAFP vs LBYL 编程模式对比
def test_eafp_vs_lbyl():
    print("\n2. EAFP vs LBYL 编程模式对比")
    
    # 测试数据
    my_dict = {'a': 1, 'b': 2, 'c': 3}
    key = 'd'
    
    print("\nLBYL (Look Before You Leap) 方式:")
    # LBYL方式：先检查再执行
    if key in my_dict:
        value = my_dict[key]
        print(f"找到键 '{key}': {value}")
    else:
        print(f"键 '{key}' 不存在")
    
    print("\nEAFP (Easier to Ask for Forgiveness than Permission) 方式:")
    # EAFP方式：直接尝试，出错再处理
    try:
        value = my_dict[key]
        print(f"找到键 '{key}': {value}")
    except KeyError:
        print(f"键 '{key}' 不存在")

# 3. 文件操作中的异常处理
def test_file_operations():
    print("\n3. 文件操作中的异常处理")
    
    # 尝试读取文件
    filename = "test_file.txt"
    
    try:
        with open(filename, 'r', encoding='utf-8') as f:
            content = f.read()
            print(f"文件内容: {content}")
    except FileNotFoundError:
        print(f"文件 '{filename}' 不存在，创建新文件")
        # 创建文件并写入内容
        with open(filename, 'w', encoding='utf-8') as f:
            f.write("这是测试文件的内容")
        print("文件创建成功")
    except PermissionError:
        print(f"没有权限访问文件 '{filename}'")
    except Exception as e:
        print(f"读取文件时发生未知错误: {e}")

# 4. 异常处理的性能考虑
import time

def test_performance():
    print("\n4. 异常处理的性能考虑")
    
    # 测试正常情况下的性能
    start_time = time.time()
    for i in range(100000):
        try:
            result = 10 / 2  # 正常执行
        except ZeroDivisionError:
            pass
    normal_time = time.time() - start_time
    
    # 测试异常情况下的性能
    start_time = time.time()
    for i in range(1000):  # 减少次数，因为异常处理较慢
        try:
            result = 10 / 0  # 会抛出异常
        except ZeroDivisionError:
            pass
    exception_time = time.time() - start_time
    
    print(f"正常执行100000次耗时: {normal_time:.4f}秒")
    print(f"异常处理1000次耗时: {exception_time:.4f}秒")
    print(f"异常处理比正常执行慢约 {(exception_time/1000)/(normal_time/100000):.1f} 倍")

if __name__ == "__main__":
    test_exception_hierarchy()
    test_eafp_vs_lbyl()
    test_file_operations()
    test_performance()