# -*- coding: utf-8 -*-
"""上下文管理器测试脚本"""

print("=== 测试1: 基本with语句 ===")
with open('sample.txt', 'r', encoding='utf-8') as f:
    print(f"文件是否关闭: {f.closed}")
    content = f.read(20)  # 只读取前20个字符
    print(f"读取内容: {repr(content)}")

print(f"with语句结束后，文件是否关闭: {f.closed}")

print("\n=== 测试2: 多文件操作 ===")
with open('sample.txt', 'r', encoding='utf-8') as input_file, \
     open('copy_output.txt', 'w', encoding='utf-8') as output_file:
    
    print("同时打开两个文件")
    content = input_file.read()
    output_file.write(f"复制的内容:\n{content}")
    print("文件复制完成")

print("验证复制结果:")
with open('copy_output.txt', 'r', encoding='utf-8') as f:
    print(repr(f.read()))

print("\n=== 测试3: 异常处理演示 ===")
try:
    with open('nonexistent.txt', 'r') as f:
        content = f.read()
except FileNotFoundError as e:
    print(f"捕获异常: {e}")
    print("即使发生异常，with语句也会正确关闭文件")

print("\n=== 测试4: 自定义上下文管理器 ===")
class FileManager:
    """自定义文件管理器"""
    def __init__(self, filename, mode):
        self.filename = filename
        self.mode = mode
        self.file = None
    
    def __enter__(self):
        print(f"打开文件: {self.filename}")
        self.file = open(self.filename, self.mode, encoding='utf-8')
        return self.file
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        print(f"关闭文件: {self.filename}")
        if self.file:
            self.file.close()
        if exc_type:
            print(f"处理异常: {exc_type.__name__}")
        return False  # 不抑制异常

# 使用自定义上下文管理器
with FileManager('custom_test.txt', 'w') as f:
    f.write('使用自定义上下文管理器写入的内容')
    print('写入完成')

# 验证结果
with open('custom_test.txt', 'r', encoding='utf-8') as f:
    print(f"自定义管理器写入的内容: {repr(f.read())}")

print("\n=== 测试5: 文件对象属性 ===")
with open('sample.txt', 'r', encoding='utf-8') as f:
    print(f"文件名: {f.name}")
    print(f"文件模式: {f.mode}")
    print(f"编码: {f.encoding}")
    print(f"是否关闭: {f.closed}")
    print(f"是否可读: {f.readable()}")
    print(f"是否可写: {f.writable()}")
    print(f"是否可定位: {f.seekable()}")

print(f"with语句外，文件是否关闭: {f.closed}")