#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
简单HTTP服务器示例
"""

import http.server
import socketserver
import urllib.parse
import json
import time
from pathlib import Path

class CustomHTTPRequestHandler(http.server.BaseHTTPRequestHandler):
    """自定义HTTP请求处理器"""
    
    def log_message(self, format, *args):
        """自定义日志格式"""
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] {self.address_string()} - {format % args}")
    
    def do_GET(self):
        """处理GET请求"""
        self.log_message(f"GET请求: {self.path}")
        
        if self.path == '/':
            self.send_homepage()
        elif self.path == '/api/info':
            self.send_api_info()
        elif self.path.startswith('/api/time'):
            self.send_current_time()
        else:
            self.send_404()
    
    def do_POST(self):
        """处理POST请求"""
        self.log_message(f"POST请求: {self.path}")
        
        if self.path == '/api/echo':
            self.handle_echo_post()
        else:
            self.send_404()
    
    def send_homepage(self):
        """发送主页"""
        html_content = """
<!DOCTYPE html>
<html>
<head>
    <title>Python HTTP服务器演示</title>
    <meta charset="utf-8">
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; }
        .container { max-width: 800px; margin: 0 auto; }
        .api-list { background: #f5f5f5; padding: 20px; border-radius: 5px; }
        .api-item { margin: 10px 0; }
        code { background: #e8e8e8; padding: 2px 5px; border-radius: 3px; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🐍 Python HTTP服务器演示</h1>
        <p>欢迎访问Python自定义HTTP服务器！</p>
        
        <h2>可用的API接口：</h2>
        <div class="api-list">
            <div class="api-item">
                <strong>GET</strong> <code>/</code> - 主页
            </div>
            <div class="api-item">
                <strong>GET</strong> <code>/api/info</code> - 服务器信息
            </div>
            <div class="api-item">
                <strong>GET</strong> <code>/api/time</code> - 当前时间
            </div>
            <div class="api-item">
                <strong>POST</strong> <code>/api/echo</code> - 回显POST数据
            </div>
        </div>
        
        <h2>测试命令：</h2>
        <pre>
# 获取服务器信息
curl http://localhost:8000/api/info

# 获取当前时间
curl http://localhost:8000/api/time

# POST数据测试
curl -X POST -d "message=Hello Server" http://localhost:8000/api/echo
        </pre>
    </div>
</body>
</html>
        """
        
        self.send_response(200)
        self.send_header('Content-type', 'text/html; charset=utf-8')
        self.end_headers()
        self.wfile.write(html_content.encode('utf-8'))
    
    def send_api_info(self):
        """发送API信息"""
        info = {
            'server': 'Python Custom HTTP Server',
            'version': '1.0',
            'timestamp': time.strftime('%Y-%m-%d %H:%M:%S'),
            'client_ip': self.client_address[0],
            'user_agent': self.headers.get('User-Agent', 'Unknown')
        }
        
        self.send_json_response(info)
    
    def send_current_time(self):
        """发送当前时间"""
        time_info = {
            'current_time': time.strftime('%Y-%m-%d %H:%M:%S'),
            'timestamp': int(time.time()),
            'timezone': time.tzname[0] if time.tzname[0] else 'UTC'
        }
        
        self.send_json_response(time_info)
    
    def handle_echo_post(self):
        """处理回显POST请求"""
        try:
            # 获取POST数据长度
            content_length = int(self.headers.get('Content-Length', 0))
            
            # 读取POST数据
            post_data = self.rfile.read(content_length)
            
            # 解析数据
            if self.headers.get('Content-Type', '').startswith('application/json'):
                # JSON数据
                data = json.loads(post_data.decode('utf-8'))
            else:
                # 表单数据
                data = urllib.parse.parse_qs(post_data.decode('utf-8'))
                # 简化单值列表
                data = {k: v[0] if len(v) == 1 else v for k, v in data.items()}
            
            response = {
                'message': '服务器收到POST数据',
                'received_data': data,
                'content_type': self.headers.get('Content-Type', 'unknown'),
                'data_size': content_length,
                'timestamp': time.strftime('%Y-%m-%d %H:%M:%S')
            }
            
            self.send_json_response(response)
            
        except Exception as e:
            error_response = {
                'error': '处理POST数据时发生错误',
                'details': str(e),
                'timestamp': time.strftime('%Y-%m-%d %H:%M:%S')
            }
            self.send_json_response(error_response, status_code=400)
    
    def send_json_response(self, data, status_code=200):
        """发送JSON响应"""
        json_data = json.dumps(data, ensure_ascii=False, indent=2)
        
        self.send_response(status_code)
        self.send_header('Content-type', 'application/json; charset=utf-8')
        self.send_header('Access-Control-Allow-Origin', '*')  # 允许跨域
        self.end_headers()
        self.wfile.write(json_data.encode('utf-8'))
    
    def send_404(self):
        """发送404错误"""
        error_response = {
            'error': '页面未找到',
            'path': self.path,
            'message': '请求的资源不存在',
            'timestamp': time.strftime('%Y-%m-%d %H:%M:%S')
        }
        
        self.send_json_response(error_response, status_code=404)

def start_server(port=8000):
    """启动HTTP服务器"""
    try:
        with socketserver.TCPServer(("", port), CustomHTTPRequestHandler) as httpd:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] HTTP服务器启动成功")
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 服务器地址: http://localhost:{port}")
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 按 Ctrl+C 停止服务器")
            print("-" * 60)
            
            httpd.serve_forever()
            
    except KeyboardInterrupt:
        print(f"\n[{time.strftime('%Y-%m-%d %H:%M:%S')}] 服务器正在关闭...")
    except Exception as e:
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 服务器启动失败: {e}")
    finally:
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 服务器已关闭")

if __name__ == '__main__':
    start_server()