#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
HTTP客户端示例 - urllib和requests库演示
"""

import urllib.request
import urllib.parse
import urllib.error
import json
import time

def urllib_demo():
    """urllib模块演示"""
    print("=" * 50)
    print("urllib模块演示")
    print("=" * 50)
    
    try:
        # GET请求示例
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 发送GET请求到httpbin.org...")
        url = 'https://httpbin.org/get'
        
        # 添加请求头
        headers = {
            'User-Agent': 'Python-urllib/3.9',
            'Accept': 'application/json'
        }
        
        request = urllib.request.Request(url, headers=headers)
        response = urllib.request.urlopen(request, timeout=10)
        
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 响应状态码: {response.getcode()}")
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 响应头信息:")
        for header, value in response.headers.items():
            print(f"  {header}: {value}")
        
        # 读取响应内容
        data = response.read().decode('utf-8')
        json_data = json.loads(data)
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 响应内容:")
        print(json.dumps(json_data, indent=2, ensure_ascii=False))
        
    except urllib.error.URLError as e:
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] URL错误: {e}")
    except Exception as e:
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 其他错误: {e}")

def urllib_post_demo():
    """urllib POST请求演示"""
    print("\n" + "=" * 50)
    print("urllib POST请求演示")
    print("=" * 50)
    
    try:
        url = 'https://httpbin.org/post'
        
        # 准备POST数据
        post_data = {
            'name': 'Python学习者',
            'message': '这是一个POST请求测试',
            'timestamp': time.strftime('%Y-%m-%d %H:%M:%S')
        }
        
        # 编码POST数据
        data = urllib.parse.urlencode(post_data).encode('utf-8')
        
        # 创建请求
        headers = {
            'Content-Type': 'application/x-www-form-urlencoded',
            'User-Agent': 'Python-urllib/3.9'
        }
        
        request = urllib.request.Request(url, data=data, headers=headers, method='POST')
        
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 发送POST请求...")
        response = urllib.request.urlopen(request, timeout=10)
        
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 响应状态码: {response.getcode()}")
        
        # 读取响应
        response_data = response.read().decode('utf-8')
        json_data = json.loads(response_data)
        
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 服务器收到的表单数据:")
        for key, value in json_data.get('form', {}).items():
            print(f"  {key}: {value}")
            
    except Exception as e:
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] POST请求错误: {e}")

def requests_demo():
    """requests库演示（如果已安装）"""
    print("\n" + "=" * 50)
    print("requests库演示")
    print("=" * 50)
    
    try:
        import requests
        
        # GET请求
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 使用requests发送GET请求...")
        response = requests.get('https://httpbin.org/get', timeout=10)
        
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 响应状态码: {response.status_code}")
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 响应头Content-Type: {response.headers.get('Content-Type')}")
        
        json_data = response.json()
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 客户端IP: {json_data.get('origin')}")
        
        # POST请求
        print(f"\n[{time.strftime('%Y-%m-%d %H:%M:%S')}] 使用requests发送POST请求...")
        post_data = {
            'username': 'python_user',
            'message': 'requests库测试消息'
        }
        
        response = requests.post('https://httpbin.org/post', data=post_data, timeout=10)
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] POST响应状态码: {response.status_code}")
        
        json_response = response.json()
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 提交的表单数据:")
        for key, value in json_response.get('form', {}).items():
            print(f"  {key}: {value}")
            
    except ImportError:
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] requests库未安装，跳过演示")
        print("安装命令: pip install requests")
    except Exception as e:
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] requests演示错误: {e}")

if __name__ == '__main__':
    print("HTTP客户端编程演示")
    print("注意: 需要网络连接才能正常运行")
    
    urllib_demo()
    urllib_post_demo()
    requests_demo()
    
    print(f"\n[{time.strftime('%Y-%m-%d %H:%M:%S')}] 演示完成")