#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
生成器演示代码
"""

import sys

# 生成器函数基本用法
def simple_generator():
    """
    简单的生成器函数
    """
    print("生成器开始执行")
    yield 1
    print("生成第一个值后继续")
    yield 2
    print("生成第二个值后继续")
    yield 3
    print("生成器执行完毕")

print("=== 生成器函数基本用法 ===")
gen = simple_generator()
print(f"生成器对象: {gen}")
print(f"第一次调用next(): {next(gen)}")
print(f"第二次调用next(): {next(gen)}")
print(f"第三次调用next(): {next(gen)}")

try:
    print(f"第四次调用next(): {next(gen)}")
except StopIteration:
    print("生成器已耗尽，抛出StopIteration异常")

# 斐波那契数列生成器
def fibonacci_generator(n):
    """
    生成斐波那契数列的生成器
    """
    a, b = 0, 1
    count = 0
    while count < n:
        yield a
        a, b = b, a + b
        count += 1

print("\n=== 斐波那契数列生成器 ===")
fib_gen = fibonacci_generator(10)
fib_list = list(fib_gen)
print(f"前10个斐波那契数: {fib_list}")

# 生成器表达式
print("\n=== 生成器表达式 ===")
squares_gen = (x**2 for x in range(5))
print(f"生成器表达式对象: {squares_gen}")
print("逐个获取值:")
for square in squares_gen:
    print(f"  {square}")

# 内存使用对比
print("\n=== 内存使用对比 ===")
# 列表推导式
list_comp = [x**2 for x in range(1000)]
list_size = sys.getsizeof(list_comp)
print(f"列表推导式内存使用: {list_size} 字节")

# 生成器表达式
gen_exp = (x**2 for x in range(1000))
gen_size = sys.getsizeof(gen_exp)
print(f"生成器表达式内存使用: {gen_size} 字节")
print(f"生成器比列表节省内存: {list_size - gen_size} 字节")

# 无限序列生成器
def infinite_counter(start=0):
    """
    无限计数器生成器
    """
    count = start
    while True:
        yield count
        count += 1

print("\n=== 无限序列生成器 ===")
counter = infinite_counter(10)
print("前5个值:")
for i, value in enumerate(counter):
    if i >= 5:
        break
    print(f"  {value}")

# 生成器的send方法
def echo_generator():
    """
    演示send方法的生成器
    """
    value = None
    while True:
        received = yield value
        if received is not None:
            value = f"Echo: {received}"
        else:
            value = "No input"

print("\n=== 生成器的send方法 ===")
echo_gen = echo_generator()
print(f"初始化生成器: {next(echo_gen)}")
print(f"发送'Hello': {echo_gen.send('Hello')}")
print(f"发送'World': {echo_gen.send('World')}")
print(f"不发送数据: {next(echo_gen)}")

# 管道处理示例
def read_numbers():
    """
    模拟读取数字的生成器
    """
    numbers = [1, 2, 3, 4, 5, 6, 7, 8, 9, 10]
    for num in numbers:
        yield num

def filter_even(numbers):
    """
    过滤偶数的生成器
    """
    for num in numbers:
        if num % 2 == 0:
            yield num

def square_numbers(numbers):
    """
    计算平方的生成器
    """
    for num in numbers:
        yield num ** 2

print("\n=== 管道处理示例 ===")
# 构建处理管道
pipeline = square_numbers(filter_even(read_numbers()))
result = list(pipeline)
print(f"管道处理结果 (偶数的平方): {result}")