#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
装饰器演示代码
"""

import time
import functools
from datetime import datetime

# 简单装饰器
def simple_decorator(func):
    """
    简单的装饰器示例
    """
    def wrapper(*args, **kwargs):
        print(f"调用函数 {func.__name__} 之前")
        result = func(*args, **kwargs)
        print(f"调用函数 {func.__name__} 之后")
        return result
    return wrapper

@simple_decorator
def greet(name):
    """
    问候函数
    """
    print(f"Hello, {name}!")
    return f"Greeting for {name}"

print("=== 简单装饰器 ===")
result = greet("Alice")
print(f"返回值: {result}")

# 计时装饰器
def timing_decorator(func):
    """
    计时装饰器
    """
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        start_time = time.time()
        result = func(*args, **kwargs)
        end_time = time.time()
        print(f"函数 {func.__name__} 执行时间: {end_time - start_time:.4f}秒")
        return result
    return wrapper

@timing_decorator
def slow_function():
    """
    模拟耗时函数
    """
    time.sleep(0.1)
    return "任务完成"

print("\n=== 计时装饰器 ===")
result = slow_function()
print(f"返回值: {result}")

# 带参数的装饰器
def repeat_decorator(times):
    """
    重复执行装饰器工厂
    """
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            results = []
            for i in range(times):
                print(f"第 {i+1} 次执行:")
                result = func(*args, **kwargs)
                results.append(result)
            return results
        return wrapper
    return decorator

@repeat_decorator(3)
def say_hello(name):
    """
    打招呼函数
    """
    message = f"你好, {name}!"
    print(f"  {message}")
    return message

print("\n=== 带参数的装饰器 ===")
results = say_hello("Bob")
print(f"所有返回值: {results}")

# 日志装饰器
def log_decorator(log_level="INFO"):
    """
    日志装饰器工厂
    """
    def decorator(func):
        @functools.wraps(func)
        def wrapper(*args, **kwargs):
            timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
            print(f"[{log_level}] {timestamp} - 调用函数 {func.__name__}")
            print(f"[{log_level}] 参数: args={args}, kwargs={kwargs}")
            try:
                result = func(*args, **kwargs)
                print(f"[{log_level}] 函数 {func.__name__} 执行成功")
                return result
            except Exception as e:
                print(f"[ERROR] 函数 {func.__name__} 执行失败: {e}")
                raise
        return wrapper
    return decorator

@log_decorator("DEBUG")
def divide(a, b):
    """
    除法函数
    """
    return a / b

print("\n=== 日志装饰器 ===")
print("正常执行:")
result = divide(10, 2)
print(f"结果: {result}")

print("\n异常执行:")
try:
    result = divide(10, 0)
except ZeroDivisionError:
    print("捕获到除零异常")

# 类装饰器
class CountCalls:
    """
    计数装饰器类
    """
    def __init__(self, func):
        self.func = func
        self.count = 0
        functools.update_wrapper(self, func)
    
    def __call__(self, *args, **kwargs):
        self.count += 1
        print(f"函数 {self.func.__name__} 被调用第 {self.count} 次")
        return self.func(*args, **kwargs)
    
    def get_count(self):
        return self.count

@CountCalls
def test_function(x):
    """
    测试函数
    """
    return x * 2

print("\n=== 类装饰器 ===")
print(f"调用1: {test_function(5)}")
print(f"调用2: {test_function(10)}")
print(f"调用3: {test_function(15)}")
print(f"总调用次数: {test_function.get_count()}")

# 缓存装饰器
def cache_decorator(func):
    """
    简单的缓存装饰器
    """
    cache = {}
    
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        # 创建缓存键
        key = str(args) + str(sorted(kwargs.items()))
        
        if key in cache:
            print(f"缓存命中: {func.__name__}{args}")
            return cache[key]
        
        print(f"计算结果: {func.__name__}{args}")
        result = func(*args, **kwargs)
        cache[key] = result
        return result
    
    wrapper.cache = cache
    return wrapper

@cache_decorator
def fibonacci(n):
    """
    斐波那契数列（递归实现）
    """
    if n <= 1:
        return n
    return fibonacci(n-1) + fibonacci(n-2)

print("\n=== 缓存装饰器 ===")
print(f"fibonacci(10) = {fibonacci(10)}")
print(f"缓存内容: {len(fibonacci.cache)} 个条目")

# 内置装饰器示例
class Person:
    """
    演示内置装饰器的类
    """
    def __init__(self, name, age):
        self._name = name
        self._age = age
    
    @property
    def name(self):
        """姓名属性"""
        return self._name
    
    @name.setter
    def name(self, value):
        if not isinstance(value, str):
            raise ValueError("姓名必须是字符串")
        self._name = value
    
    @property
    def age(self):
        """年龄属性"""
        return self._age
    
    @age.setter
    def age(self, value):
        if not isinstance(value, int) or value < 0:
            raise ValueError("年龄必须是非负整数")
        self._age = value
    
    @staticmethod
    def is_adult(age):
        """静态方法：判断是否成年"""
        return age >= 18
    
    @classmethod
    def from_string(cls, person_str):
        """类方法：从字符串创建Person对象"""
        name, age = person_str.split('-')
        return cls(name, int(age))
    
    def __str__(self):
        return f"Person(name='{self.name}', age={self.age})"

print("\n=== 内置装饰器示例 ===")
person = Person("Alice", 25)
print(f"创建对象: {person}")
print(f"访问属性: name={person.name}, age={person.age}")

# 修改属性
person.name = "Bob"
person.age = 30
print(f"修改后: {person}")

# 静态方法
print(f"25岁是否成年: {Person.is_adult(25)}")
print(f"15岁是否成年: {Person.is_adult(15)}")

# 类方法
person2 = Person.from_string("Charlie-35")
print(f"从字符串创建: {person2}")