#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
上下文管理器演示代码
"""

import contextlib
import time
import os
from contextlib import contextmanager, suppress, ExitStack

# 自定义上下文管理器类
class TimerContext:
    """
    计时上下文管理器
    """
    def __init__(self, name="操作"):
        self.name = name
        self.start_time = None
    
    def __enter__(self):
        print(f"开始执行 {self.name}")
        self.start_time = time.time()
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        end_time = time.time()
        duration = end_time - self.start_time
        
        if exc_type is None:
            print(f"{self.name} 执行完成，耗时: {duration:.4f}秒")
        else:
            print(f"{self.name} 执行异常，耗时: {duration:.4f}秒")
            print(f"异常类型: {exc_type.__name__}")
            print(f"异常信息: {exc_val}")
        
        # 返回False表示不抑制异常
        return False

print("=== 自定义上下文管理器类 ===")
with TimerContext("数据处理"):
    time.sleep(0.1)
    result = sum(range(1000000))
    print(f"计算结果: {result}")

print("\n异常情况:")
try:
    with TimerContext("异常操作"):
        time.sleep(0.05)
        raise ValueError("模拟异常")
except ValueError as e:
    print(f"捕获异常: {e}")

# 文件操作上下文管理器
class FileManager:
    """
    文件管理上下文管理器
    """
    def __init__(self, filename, mode='r', encoding='utf-8'):
        self.filename = filename
        self.mode = mode
        self.encoding = encoding
        self.file = None
    
    def __enter__(self):
        print(f"打开文件: {self.filename}")
        self.file = open(self.filename, self.mode, encoding=self.encoding)
        return self.file
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        if self.file:
            print(f"关闭文件: {self.filename}")
            self.file.close()
        
        if exc_type is not None:
            print(f"文件操作异常: {exc_val}")
        
        return False

print("\n=== 文件操作上下文管理器 ===")
test_file = "test_context.txt"

# 写入文件
with FileManager(test_file, 'w') as f:
    f.write("Hello, Context Manager!\n")
    f.write("这是测试内容。\n")

# 读取文件
with FileManager(test_file, 'r') as f:
    content = f.read()
    print(f"文件内容:\n{content}")

# 清理测试文件
if os.path.exists(test_file):
    os.remove(test_file)
    print(f"删除测试文件: {test_file}")

# 使用@contextmanager装饰器
@contextmanager
def database_transaction():
    """
    模拟数据库事务上下文管理器
    """
    print("开始数据库事务")
    try:
        # 模拟数据库连接
        db_connection = {"connected": True, "transaction_active": True}
        yield db_connection
        print("提交事务")
    except Exception as e:
        print(f"回滚事务，原因: {e}")
        raise
    finally:
        print("关闭数据库连接")

print("\n=== @contextmanager装饰器 ===")
print("正常事务:")
with database_transaction() as db:
    print(f"执行数据库操作: {db}")
    # 模拟数据库操作
    print("插入数据...")
    print("更新数据...")

print("\n异常事务:")
try:
    with database_transaction() as db:
        print(f"执行数据库操作: {db}")
        print("插入数据...")
        raise Exception("数据库操作失败")
except Exception as e:
    print(f"处理异常: {e}")

# 临时目录上下文管理器
@contextmanager
def temporary_directory(prefix="temp_"):
    """
    临时目录上下文管理器
    """
    import tempfile
    import shutil
    
    temp_dir = tempfile.mkdtemp(prefix=prefix)
    print(f"创建临时目录: {temp_dir}")
    
    try:
        yield temp_dir
    finally:
        print(f"删除临时目录: {temp_dir}")
        shutil.rmtree(temp_dir, ignore_errors=True)

print("\n=== 临时目录上下文管理器 ===")
with temporary_directory("demo_") as temp_dir:
    print(f"在临时目录中工作: {temp_dir}")
    
    # 在临时目录中创建文件
    temp_file = os.path.join(temp_dir, "test.txt")
    with open(temp_file, 'w') as f:
        f.write("临时文件内容")
    
    print(f"创建临时文件: {temp_file}")
    print(f"文件是否存在: {os.path.exists(temp_file)}")

# contextlib.suppress示例
print("\n=== contextlib.suppress示例 ===")

# 不使用suppress
print("不使用suppress:")
try:
    int("not_a_number")
except ValueError:
    print("捕获到ValueError异常")

# 使用suppress
print("使用suppress:")
with suppress(ValueError):
    int("not_a_number")
    print("这行不会执行")
print("异常被抑制，程序继续执行")

# 多个异常类型
print("\n抑制多种异常类型:")
with suppress(ValueError, TypeError, KeyError):
    # 这些异常都会被抑制
    data = {"key": "value"}
    result = data["nonexistent_key"]  # KeyError
print("KeyError被抑制")

# ExitStack示例
print("\n=== ExitStack示例 ===")

@contextmanager
def managed_resource(name):
    """
    模拟资源管理
    """
    print(f"获取资源: {name}")
    try:
        yield name
    finally:
        print(f"释放资源: {name}")

# 使用ExitStack管理多个上下文
with ExitStack() as stack:
    # 动态添加上下文管理器
    resource1 = stack.enter_context(managed_resource("数据库连接"))
    resource2 = stack.enter_context(managed_resource("文件句柄"))
    resource3 = stack.enter_context(managed_resource("网络连接"))
    
    print(f"使用资源: {resource1}, {resource2}, {resource3}")
    
    # 可以根据条件添加更多资源
    if True:  # 某个条件
        resource4 = stack.enter_context(managed_resource("缓存连接"))
        print(f"额外资源: {resource4}")

print("所有资源已自动释放")

# 嵌套上下文管理器
class ResourcePool:
    """
    资源池上下文管理器
    """
    def __init__(self, pool_name, size=3):
        self.pool_name = pool_name
        self.size = size
        self.resources = []
    
    def __enter__(self):
        print(f"初始化资源池: {self.pool_name}")
        for i in range(self.size):
            resource = f"{self.pool_name}_resource_{i+1}"
            self.resources.append(resource)
            print(f"  创建资源: {resource}")
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        print(f"清理资源池: {self.pool_name}")
        for resource in self.resources:
            print(f"  销毁资源: {resource}")
        self.resources.clear()
        return False
    
    def get_resource(self):
        if self.resources:
            return self.resources[0]
        return None

print("\n=== 嵌套上下文管理器 ===")
with ResourcePool("数据库池", 2) as db_pool:
    with ResourcePool("缓存池", 2) as cache_pool:
        print(f"使用数据库资源: {db_pool.get_resource()}")
        print(f"使用缓存资源: {cache_pool.get_resource()}")
        
        # 模拟一些操作
        print("执行业务逻辑...")

# 条件上下文管理器
@contextmanager
def conditional_context(condition, context_manager):
    """
    条件上下文管理器
    """
    if condition:
        with context_manager as value:
            yield value
    else:
        yield None

print("\n=== 条件上下文管理器 ===")
for use_timer in [True, False]:
    print(f"使用计时器: {use_timer}")
    with conditional_context(use_timer, TimerContext("条件操作")) as timer:
        if timer:
            print("  在计时器上下文中执行")
        else:
            print("  直接执行，无计时器")
        time.sleep(0.02)
    print()