#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
闭包演示代码
"""

# 基本闭包示例
def outer_function(x):
    """
    外层函数
    """
    def inner_function(y):
        """
        内层函数，形成闭包
        """
        return x + y  # 引用外层函数的变量x
    return inner_function

print("=== 基本闭包示例 ===")
closure1 = outer_function(10)
closure2 = outer_function(20)

print(f"closure1(5) = {closure1(5)}")  # 10 + 5 = 15
print(f"closure2(5) = {closure2(5)}")  # 20 + 5 = 25
print(f"closure1和closure2是不同的函数对象: {closure1 is not closure2}")

# 闭包变量的持久化
def make_counter():
    """
    创建计数器闭包
    """
    count = 0
    
    def counter():
        nonlocal count
        count += 1
        return count
    
    return counter

print("\n=== 闭包变量的持久化 ===")
counter1 = make_counter()
counter2 = make_counter()

print("counter1调用:")
print(f"  第1次: {counter1()}")
print(f"  第2次: {counter1()}")
print(f"  第3次: {counter1()}")

print("counter2调用:")
print(f"  第1次: {counter2()}")
print(f"  第2次: {counter2()}")

print("counter1再次调用:")
print(f"  第4次: {counter1()}")

# 闭包实现装饰器
def timing_decorator_closure(func):
    """
    使用闭包实现的计时装饰器
    """
    import time
    
    def wrapper(*args, **kwargs):
        start_time = time.time()
        result = func(*args, **kwargs)
        end_time = time.time()
        print(f"函数 {func.__name__} 执行时间: {end_time - start_time:.4f}秒")
        return result
    
    return wrapper

@timing_decorator_closure
def test_function():
    """
    测试函数
    """
    import time
    time.sleep(0.05)
    return "执行完成"

print("\n=== 闭包实现装饰器 ===")
result = test_function()
print(f"返回值: {result}")

# 函数工厂
def make_multiplier(n):
    """
    创建乘法器函数工厂
    """
    def multiplier(x):
        return x * n
    return multiplier

print("\n=== 函数工厂 ===")
double = make_multiplier(2)
triple = make_multiplier(3)
quadruple = make_multiplier(4)

print(f"double(5) = {double(5)}")
print(f"triple(5) = {triple(5)}")
print(f"quadruple(5) = {quadruple(5)}")

# 配置化函数
def make_validator(min_length, max_length):
    """
    创建字符串验证器
    """
    def validator(text):
        if not isinstance(text, str):
            return False, "输入必须是字符串"
        
        length = len(text)
        if length < min_length:
            return False, f"长度不能少于{min_length}个字符"
        if length > max_length:
            return False, f"长度不能超过{max_length}个字符"
        
        return True, "验证通过"
    
    return validator

print("\n=== 配置化函数 ===")
username_validator = make_validator(3, 20)
password_validator = make_validator(8, 50)

test_cases = [
    ("ab", "用户名"),
    ("alice", "用户名"),
    ("a" * 25, "用户名"),
    ("123", "密码"),
    ("password123", "密码"),
]

for text, field_type in test_cases:
    if field_type == "用户名":
        is_valid, message = username_validator(text)
    else:
        is_valid, message = password_validator(text)
    
    print(f"  {field_type} '{text}': {'✓' if is_valid else '✗'} {message}")

# 事件处理闭包
def make_event_handler(event_type):
    """
    创建事件处理器
    """
    handlers = []
    
    def add_handler(handler):
        handlers.append(handler)
        print(f"为 {event_type} 事件添加处理器: {handler.__name__}")
    
    def trigger_event(data):
        print(f"触发 {event_type} 事件，数据: {data}")
        for handler in handlers:
            handler(data)
    
    def get_handler_count():
        return len(handlers)
    
    # 返回多个函数
    return add_handler, trigger_event, get_handler_count

print("\n=== 事件处理闭包 ===")

# 创建点击事件处理器
add_click_handler, trigger_click, get_click_count = make_event_handler("click")

# 定义事件处理函数
def on_button_click(data):
    print(f"  按钮点击处理: {data}")

def on_link_click(data):
    print(f"  链接点击处理: {data}")

# 添加处理器
add_click_handler(on_button_click)
add_click_handler(on_link_click)

print(f"当前有 {get_click_count()} 个点击事件处理器")

# 触发事件
trigger_click("用户点击了提交按钮")

# 闭包的作用域链
def level1(a):
    """
    第一层函数
    """
    def level2(b):
        """
        第二层函数
        """
        def level3(c):
            """
            第三层函数，可以访问所有外层变量
            """
            return f"a={a}, b={b}, c={c}, sum={a+b+c}"
        return level3
    return level2

print("\n=== 闭包的作用域链 ===")
func = level1(10)(20)
result = func(30)
print(f"多层嵌套闭包结果: {result}")

# nonlocal关键字示例
def make_accumulator(initial=0):
    """
    创建累加器，演示nonlocal的使用
    """
    total = initial
    
    def add(value):
        nonlocal total
        total += value
        return total
    
    def subtract(value):
        nonlocal total
        total -= value
        return total
    
    def get_total():
        return total
    
    def reset(new_value=0):
        nonlocal total
        old_total = total
        total = new_value
        return old_total
    
    return add, subtract, get_total, reset

print("\n=== nonlocal关键字示例 ===")
add, subtract, get_total, reset = make_accumulator(100)

print(f"初始值: {get_total()}")
print(f"加10: {add(10)}")
print(f"加20: {add(20)}")
print(f"减5: {subtract(5)}")
print(f"当前总计: {get_total()}")
print(f"重置前的值: {reset(0)}")
print(f"重置后的值: {get_total()}")

# 闭包陷阱示例
print("\n=== 闭包陷阱示例 ===")

# 错误的方式
print("错误的方式:")
functions_wrong = []
for i in range(3):
    functions_wrong.append(lambda x: x + i)

for j, func in enumerate(functions_wrong):
    print(f"  函数{j}(10) = {func(10)}")  # 都会输出 12，因为i的最终值是2

# 正确的方式
print("正确的方式:")
functions_correct = []
for i in range(3):
    functions_correct.append(lambda x, i=i: x + i)  # 使用默认参数捕获i的值

for j, func in enumerate(functions_correct):
    print(f"  函数{j}(10) = {func(10)}")

# 另一种正确的方式
print("另一种正确的方式:")
def make_adder(n):
    return lambda x: x + n

functions_correct2 = [make_adder(i) for i in range(3)]
for j, func in enumerate(functions_correct2):
    print(f"  函数{j}(10) = {func(10)}")