#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
异步网络编程示例 - asyncio
"""

import asyncio
import time
import json

class AsyncTCPServer:
    """异步TCP服务器"""
    
    def __init__(self, host='localhost', port=8888):
        self.host = host
        self.port = port
        self.clients = set()
    
    async def handle_client(self, reader, writer):
        """处理客户端连接"""
        client_address = writer.get_extra_info('peername')
        self.clients.add(writer)
        
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 异步服务器: 客户端 {client_address} 已连接")
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 当前连接数: {len(self.clients)}")
        
        try:
            while True:
                # 异步读取数据
                data = await reader.read(1024)
                if not data:
                    break
                
                message = data.decode('utf-8').strip()
                print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 收到来自 {client_address} 的消息: {message}")
                
                # 处理特殊命令
                if message.lower() == 'time':
                    response = f"当前时间: {time.strftime('%Y-%m-%d %H:%M:%S')}"
                elif message.lower() == 'clients':
                    response = f"当前连接客户端数量: {len(self.clients)}"
                elif message.lower().startswith('broadcast:'):
                    # 广播消息给所有客户端
                    broadcast_msg = message[10:]  # 去掉 'broadcast:' 前缀
                    await self.broadcast_message(f"广播消息: {broadcast_msg}", exclude=writer)
                    response = "广播消息已发送"
                else:
                    response = f"异步服务器收到: {message}"
                
                # 异步发送响应
                writer.write(response.encode('utf-8'))
                await writer.drain()
                
        except asyncio.CancelledError:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 客户端 {client_address} 连接被取消")
        except Exception as e:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 处理客户端 {client_address} 时发生错误: {e}")
        finally:
            self.clients.discard(writer)
            writer.close()
            await writer.wait_closed()
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 客户端 {client_address} 已断开连接")
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 当前连接数: {len(self.clients)}")
    
    async def broadcast_message(self, message, exclude=None):
        """向所有客户端广播消息"""
        if not self.clients:
            return
        
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 广播消息给 {len(self.clients)} 个客户端: {message}")
        
        # 创建发送任务列表
        tasks = []
        for client in self.clients.copy():  # 使用副本避免迭代时修改
            if client != exclude:
                tasks.append(self.send_to_client(client, message))
        
        # 并发发送消息
        if tasks:
            await asyncio.gather(*tasks, return_exceptions=True)
    
    async def send_to_client(self, writer, message):
        """向单个客户端发送消息"""
        try:
            writer.write(message.encode('utf-8'))
            await writer.drain()
        except Exception as e:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 发送消息到客户端失败: {e}")
            self.clients.discard(writer)
    
    async def start_server(self):
        """启动异步服务器"""
        server = await asyncio.start_server(
            self.handle_client, 
            self.host, 
            self.port
        )
        
        addr = server.sockets[0].getsockname()
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 异步TCP服务器启动，监听 {addr[0]}:{addr[1]}")
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 支持的命令: time, clients, broadcast:消息内容")
        print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 按 Ctrl+C 停止服务器")
        
        async with server:
            await server.serve_forever()

class AsyncTCPClient:
    """异步TCP客户端"""
    
    def __init__(self, host='localhost', port=8888):
        self.host = host
        self.port = port
        self.reader = None
        self.writer = None
    
    async def connect(self):
        """连接到服务器"""
        try:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 正在连接异步服务器 {self.host}:{self.port}...")
            self.reader, self.writer = await asyncio.open_connection(self.host, self.port)
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 已连接到异步服务器")
            return True
        except Exception as e:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 连接失败: {e}")
            return False
    
    async def send_message(self, message):
        """发送消息"""
        if not self.writer:
            print("未连接到服务器")
            return None
        
        try:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 发送消息: {message}")
            self.writer.write(message.encode('utf-8'))
            await self.writer.drain()
            
            # 接收响应
            data = await self.reader.read(1024)
            response = data.decode('utf-8')
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 服务器响应: {response}")
            return response
            
        except Exception as e:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 发送消息失败: {e}")
            return None
    
    async def close(self):
        """关闭连接"""
        if self.writer:
            self.writer.close()
            await self.writer.wait_closed()
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 客户端连接已关闭")

async def run_client_demo():
    """运行客户端演示"""
    client = AsyncTCPClient()
    
    if await client.connect():
        # 发送测试消息
        messages = [
            "Hello Async Server!",
            "time",
            "clients",
            "这是异步客户端测试",
            "broadcast:大家好，这是广播消息！",
            "再见"
        ]
        
        for message in messages:
            await client.send_message(message)
            await asyncio.sleep(1)  # 异步等待
        
        await client.close()

async def run_multiple_clients(num_clients=3):
    """运行多个客户端演示并发连接"""
    print(f"\n[{time.strftime('%Y-%m-%d %H:%M:%S')}] 启动 {num_clients} 个并发客户端...")
    
    async def client_task(client_id):
        client = AsyncTCPClient()
        if await client.connect():
            await client.send_message(f"客户端{client_id}: Hello!")
            await asyncio.sleep(2)
            await client.send_message(f"客户端{client_id}: 测试并发连接")
            await client.close()
    
    # 并发运行多个客户端
    tasks = [client_task(i) for i in range(1, num_clients + 1)]
    await asyncio.gather(*tasks)
    
    print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 所有客户端任务完成")

def main():
    """主函数"""
    import sys
    
    if len(sys.argv) > 1 and sys.argv[1] == 'client':
        # 运行客户端
        print("异步TCP客户端演示")
        asyncio.run(run_client_demo())
    elif len(sys.argv) > 1 and sys.argv[1] == 'multi-client':
        # 运行多客户端演示
        print("多客户端并发演示")
        asyncio.run(run_multiple_clients())
    else:
        # 运行服务器
        print("异步TCP服务器演示")
        server = AsyncTCPServer()
        try:
            asyncio.run(server.start_server())
        except KeyboardInterrupt:
            print(f"\n[{time.strftime('%Y-%m-%d %H:%M:%S')}] 服务器正在关闭...")
        except Exception as e:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 服务器错误: {e}")
        finally:
            print(f"[{time.strftime('%Y-%m-%d %H:%M:%S')}] 异步服务器已关闭")

if __name__ == '__main__':
    main()